import { Injectable } from '@nestjs/common';
import dict from '~constants/dict';
import { HandledError } from '~core/classes/handled-error';
import { UserEntity } from '~db/entities/user.entity';
import { db } from '~db/repositories';
import { QueryModel } from '~model/query';

@Injectable()
export class AdvertisService<T> {
  async list(tagName: string, _query: QueryModel, user: UserEntity) {
    const { offset = 1, size = 10 } = _query || {};
    const tag = tagName
      .split(/\.?(?=[A-Z])/)
      .join(' ')
      .toLocaleString()
      .split(' ')[0];

    const repo = await db.getRepository(tagName);

    let query = repo
      .createQueryBuilder(tag)
      .leftJoinAndSelect(`${tag}.advertizer`, 'advertizer')
      .where('advertizer.id = :id', { id: user.id });

    let [list, total] = await Promise.all([
      query.paginate({ offset, size }).getMany(),
      query.getCount(),
    ]);

    return list;
  }

  async single(tagName, id) {
    const repo = await db.getRepository(tagName);

    const item = await repo.findOneBy({ id });

    if (!item) throw new HandledError(dict.RECORDE_DOES_NOT_EkXIST);

    return item;
  }

  async upsert<T>(tagName: string, dto: any, user: UserEntity) {
    const { id, ...rest } = dto || {};
    const repo = await db.getRepository(tagName);

    if (id) {
      //update
      let item = await repo.findOneBy({ id });
      if (!item) throw new HandledError(dict.RECORDE_DOES_NOT_EkXIST);

      item.update({ ...rest });
      await item.save();
    } else {
      //insert
      const item = await repo.save({ ...rest, advertizer: user });
    }
    return dict.SUCCESS_MESSAGE;
  }

  async remove(tagName: string, id: string) {
    const repo = await db.getRepository(tagName);

    const item = await repo.findOneBy({ id });

    if (!item) throw new HandledError(dict.RECORDE_DOES_NOT_EkXIST);

    await item.softRemove();

    return dict.SUCCESS_MESSAGE;
  }
}
