import { ApiProperty } from '@nestjs/swagger';
import { CommandHandler, ICommandHandler } from '@nestjs/cqrs';
import { db } from '~db/repositories';
import Dict from '~constants/dict';
import { PlaceEntity } from '~db/entities/place.entity';
import { In, Not } from 'typeorm';
import { LocationAddressModel } from '~model/address';
import { UserEntity } from '~db/entities/user.entity';
import { ConsultantEntity } from '~db/entities/user/consultant.entity';
import { IsArray, IsString } from 'class-validator';
import { HandledError } from '~core/classes/handled-error';

const consultants = db.getRepository(ConsultantEntity);
const places = db.getRepository(PlaceEntity);

export class UpsertConsultantUserCommand {
  constructor(
    public dto: UpsertConsultantUserRequestModel,
    public user: UserEntity
  ) {}
}

@CommandHandler(UpsertConsultantUserCommand)
export class UpsertConsultantHandler
  implements ICommandHandler<UpsertConsultantUserCommand>
{
  async execute({
    dto,
    user: current,
  }: UpsertConsultantUserCommand): Promise<any> {
    const { cityId, areasId, ...rest } = dto;

    const city = await places.findOneBy({ id: cityId });
    const areas = await places.findBy({ id: In(areasId) });

    if (current.subscriberType === 'none') {
      current.subscriberType = 'consultant';
      // check uniq userName

      const userNameExist = await consultants.findOneBy({
        userName: dto.userName,
      });
      if (userNameExist) throw new HandledError(Dict.USERNAME_EXIST);

      await db.manager.update(UserEntity, { id: current.id }, {
        ...rest,
        city,
        areas,
        subscriberType: 'consultant',
        type: 'ConsultantEntity',
      } as any);
    } else {
      //insert
      const userNameExist = await consultants.findOneBy({
        id: Not(current.id),
        userName: dto.userName,
      });
      if (userNameExist) throw new HandledError(Dict.USERNAME_EXIST);

      const consultant = await consultants.findOneBy({ id: current.id });
      consultant.update({ ...rest, city, areas });
      await consultant.save();
    }
    return Dict.SUCCESS_MESSAGE;
  }
}

export class UpsertConsultantUserRequestModel {
  @IsString()
  @ApiProperty()
  firstName: string;

  @IsString()
  @ApiProperty()
  lastName: string;

  @IsString()
  @ApiProperty()
  userName: string;

  @IsString()
  @ApiProperty()
  licenseImg: string;

  @ApiProperty({ type: LocationAddressModel })
  address: LocationAddressModel;

  @IsString()
  @ApiProperty()
  nationalCode: string;

  @IsString()
  @ApiProperty()
  cityId: string;

  @IsArray()
  @ApiProperty({ type: [String] })
  areasId: string[];
}
