import { Body, Controller, Get, Patch, Post, UseGuards } from '@nestjs/common';
import { ApiBearerAuth, ApiTags } from '@nestjs/swagger';
import { CommandBus, QueryBus } from '@nestjs/cqrs';
import {
  UpsertAgencyUserCommand,
  UpsertAgencyUserRequestModel,
  UpsertConsultantUserCommand,
  UpsertConsultantUserRequestModel,
  UpsertPersonalUserCommand,
  UpsertPersonalUserRequestModel,
} from '~modules/users/handlers';
import { CurrentUser } from '~core/decorators/current-user.decorator';
import { JwtAuthGuard } from '~core/guards/jwt-guard';
import { UserEntity } from '~db/entities/user.entity';

@UseGuards(JwtAuthGuard)
@ApiTags('User')
@Controller('user')
@ApiBearerAuth('access-token')
export class UserController {
  constructor(
    private readonly command: CommandBus,
    private readonly queryBus: QueryBus
  ) {}

  @Post('agency')
  async upsertAgency(
    @Body() data: UpsertAgencyUserRequestModel,
    @CurrentUser() user: UserEntity
  ) {
    return await this.command.execute(new UpsertAgencyUserCommand(data, user));
  }

  @Post('consultant')
  async upsertConsultant(
    @Body() data: UpsertConsultantUserRequestModel,
    @CurrentUser() user: UserEntity
  ) {
    return await this.command.execute(
      new UpsertConsultantUserCommand(data, user)
    );
  }

  @Post('personal')
  async upsertPersonal(
    @Body() data: UpsertPersonalUserRequestModel,
    @CurrentUser() user: UserEntity
  ) {
    return await this.command.execute(
      new UpsertPersonalUserCommand(data, user)
    );
  }

  @Get('')
  async getCurrentUer(@CurrentUser() user: UserEntity) {
    return user;
  }
}
