import moment from 'moment-timezone';
import { Brackets } from 'typeorm';
import { SelectQueryBuilder } from 'typeorm/query-builder/SelectQueryBuilder';

interface IPaginateProps {
  offset: any;
  size: any;
}

declare module 'typeorm/query-builder/SelectQueryBuilder' {
  interface SelectQueryBuilder<Entity> {
    regex(
      this: SelectQueryBuilder<Entity>,
      fields: Array<string>,
      value?: any
    ): SelectQueryBuilder<Entity>;

    paginate(
      this: SelectQueryBuilder<Entity>,
      { offset, size }: IPaginateProps
    ): SelectQueryBuilder<Entity>;

    dateRange(
      this: SelectQueryBuilder<Entity>,
      key: string,
      { form, to }: any
    ): SelectQueryBuilder<Entity>;

    active(
      this: SelectQueryBuilder<Entity>,
      key: string
    ): SelectQueryBuilder<Entity>;
  }
}

SelectQueryBuilder.prototype.regex = function <Entity>(
  this: SelectQueryBuilder<any>,
  fields: Array<string>,
  value?: any
): SelectQueryBuilder<any> {
  if (!!value) {
    this.andWhere(
      new Brackets((qb) => {
        fields.forEach((x) => {
          qb.orWhere(`${x} like :value`, { value: `%${value}%` });
        });
      })
    );
  }
  return this;
};

SelectQueryBuilder.prototype.paginate = function <Entity>(
  this: SelectQueryBuilder<any>,
  { offset = 1, size = 10 }: IPaginateProps
): SelectQueryBuilder<any> {
  const limit = Number(size);
  const skip = (Number(offset) - 1) * Number(limit);
  this.take(size).skip(skip);
  return this;
};

SelectQueryBuilder.prototype.dateRange = function <Entity>(
  this: SelectQueryBuilder<any>,

  key: string,
  { fromDate, toDate }: any
): SelectQueryBuilder<any> {
  if (fromDate && !toDate) {
    const from = moment(fromDate).toISOString();
    this.andWhere(`${key} > :from`, { from });
  }

  if (toDate && !fromDate) {
    const to = moment(toDate).toISOString();
    this.andWhere(`${key} <= :to`, { to });
  }

  if (toDate && fromDate) {
    const from = moment(fromDate).toISOString();
    const to = moment(toDate).toISOString();
    this.andWhere(`${key} <= :to AND ${key} > :from`, {
      to,
      from,
    });
  }
  return this;
};

SelectQueryBuilder.prototype.active = function <Entity>(
  this: SelectQueryBuilder<any>,
  key: string
): SelectQueryBuilder<any> {
  this.andWhere(`${key}.active = true`);
  return this;
};