import {
  Body,
  Controller,
  Delete,
  Get,
  NestInterceptor,
  Param,
  PipeTransform,
  Post,
  Query,
  Type,
  UseGuards,
  UseInterceptors,
} from '@nestjs/common';
import { ApiBearerAuth, ApiBody, ApiOkResponse, ApiProperty, ApiResponse, ApiTags } from '@nestjs/swagger';
import { AdvertisService } from './advers.service';
import { CurrentUser } from '~core/decorators/current-user.decorator';
import { JwtAuthGuard } from '~core/guards/jwt-guard';
import { UserEntity } from '~db/entities/user.entity';
import { AdversPipe } from './advers.pipe';
import { AdversInterceptor } from './advers.interceptor';

export function AdvertiseControllerMixin<T>(
  entityClass: new () => T,
  pipe?: Type<PipeTransform> | PipeTransform,
  interceptor?: Type<NestInterceptor> | NestInterceptor
) {
  const tagName = entityClass.name;

  class ListResponse {
    @ApiProperty()
    status: boolean;

    @ApiProperty({ type: [entityClass] })
    data: any[];
  }

  class MessageResponse {
    @ApiProperty()
    status: boolean;

    @ApiProperty({ type: String })
    message: string;
  }

  @ApiTags(tagName.split(/\.?(?=[A-Z])/).join(' '))
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth('access-token')
  class GenericController {
    constructor(public readonly srv: AdvertisService<T>) {}

    @Get(':id')
    @ApiOkResponse({ type: entityClass})
    @UseInterceptors(interceptor || AdversInterceptor)
    async signle(@Param('id') id: string) {
      return await this.srv.single(tagName, id);
    }

    @Post()
    @ApiBody({ type: entityClass })
    @ApiResponse({type: MessageResponse})
    async upsert(@Body(pipe || AdversPipe ) dto: T, @CurrentUser() user: UserEntity) {
      return await this.srv.upsert(tagName, dto, user);
    }

    @Delete(':id')
    @ApiResponse({type: MessageResponse})
    async remove(@Param('id') id: string, @CurrentUser() user: UserEntity) {
      return await this.srv.remove(tagName, id);
    }
  }

  Controller(
    `advertise/${tagName
      .split(/\.?(?=[A-Z])/)
      .join('-')
      .toLowerCase()}`
  )(GenericController);
  return GenericController;
}
