import { ApiProperty } from '@nestjs/swagger';
import {
  IPlaceEntity,
  PlaceEntity,
  PlacesType,
} from '~db/entities/place.entity';
import {
  IPlaceInfoEntity,
  PlaceInfoEntity,
} from '~db/entities/place-info.entity';
import { CommandHandler, ICommandHandler } from '@nestjs/cqrs';
import { db } from '~db/repositories';
import { HandledError } from '~core/classes/handled-error';
import Dict from '~constants/dict';
import { IsNotEmpty } from 'class-validator';
import { Not } from 'typeorm';

const _places = db.getRepository(PlaceEntity);
const _placesInfo = db.getRepository(PlaceInfoEntity);

export class UpsertPlaceCommand {
  constructor(public data: UpsertPlaceRequest) {}
}

@CommandHandler(UpsertPlaceCommand)
export class UpsertPlaceHandler implements ICommandHandler<UpsertPlaceCommand> {
  async execute({ data }: UpsertPlaceCommand): Promise<any> {
    const { id, name, eName, type, parent: _parentId, ...rest } = data || {};

    if (id) {
      //update
      let place = await _places.findOneBy({ eName, id: Not(`${id}`) });
      if (place) throw new HandledError(Dict.ENAME_IS_USED);

      place = await _places.findOne({
        where: { id },
        relations: {
          info: true,
        },
      });
      if (!place) throw new HandledError(Dict.RECORDE_DOES_NOT_EkXIST);

      place.update({ name, eName, type });
      if (_parentId) {
        let parent = await _places.findOneBy({ id: _parentId });

        if (!parent) throw new HandledError(Dict.RECORDE_DOES_NOT_EkXIST);

        place.parent = parent;
      } else {
        place.parent = null;
      }
      await place.save();

      const info = await _placesInfo.findOneBy({ id: place.info.id });
      info.update({ ...rest });
      await info.save();

      return Dict.SUCCESS_MESSAGE;
    } else {
      // insert
      let place = await _places.findOneBy({ eName });

      if (place) throw new HandledError(Dict.ENAME_IS_USED);

      const placeInfo = new PlaceInfoEntity({
        ...rest,
      });

      await placeInfo.save();

      place = new PlaceEntity({
        name,
        eName,
        type,
        info: placeInfo,
      });

      if (_parentId) {
        let parent = await _places.findOneBy({ id: _parentId });

        if (!parent) throw new HandledError(Dict.RECORDE_DOES_NOT_EkXIST);

        place.parent = parent;
      }

      await place.save();

      return Dict.SUCCESS_MESSAGE;
    }
  }
}

export class UpsertPlaceRequest
  implements Omit<IPlaceEntity, 'parent'>, IPlaceInfoEntity
{
  @ApiProperty()
  id: string;

  @ApiProperty()
  parent: string;

  @IsNotEmpty({ message: Dict.BAD_REQUEST })
  @ApiProperty()
  name: string;

  @IsNotEmpty({ message: Dict.BAD_REQUEST })
  @ApiProperty()
  eName: string;

  @IsNotEmpty({ message: Dict.BAD_REQUEST })
  @ApiProperty({ enum: ['country', 'province'] })
  type: PlacesType;

  @ApiProperty({ type: Boolean })
  canInsertAdvertise: boolean;

  @ApiProperty({ type: Boolean })
  canInsertCommercial: boolean;

  @ApiProperty({ type: Boolean })
  canInsertSpecialAdvertise: boolean;

  @ApiProperty({ type: [String] })
  closeCities: string[];

  @ApiProperty({ type: Boolean })
  isCapitalCounty: boolean;

  @ApiProperty()
  location: string;

  @ApiProperty()
  polygon: string;

  @ApiProperty({ type: Boolean })
  showAdvertise: boolean;

  @ApiProperty({ type: Boolean })
  showCommercial: boolean;

  @ApiProperty({ type: Boolean })
  showSpecialAdvertise: boolean;
}
