import { ApiProperty } from '@nestjs/swagger';
import { CommandHandler, ICommandHandler } from '@nestjs/cqrs';
import { db } from '~db/repositories';
import Dict from '~constants/dict';
import { PlaceEntity } from '~db/entities/place.entity';
import { In, Not } from 'typeorm';
import { LocationAddressModel } from '~model/address';
import { HandledError } from '~core/classes/handled-error';
import { UserEntity } from '~db/entities/user.entity';
import { AgencyEntity } from '~db/entities/user/agency.entity';
import { IsArray, IsOptional, IsString } from 'class-validator';

const agencies = db.getRepository(AgencyEntity);
const places = db.getRepository(PlaceEntity);

export class UpsertAgencyUserCommand {
  constructor(
    public dto: UpsertAgencyUserRequestModel,
    public user: UserEntity
  ) {}
}

@CommandHandler(UpsertAgencyUserCommand)
export class UpsertAgencyUserHandler
  implements ICommandHandler<UpsertAgencyUserCommand>
{
  async execute({ user: current, dto }: UpsertAgencyUserCommand): Promise<any> {
    const { cityId, areasId, title, userName, ...rest } = dto;
    const city = await places.findOneBy({ id: cityId });
    const areas = await places.findBy({
      id: In(areasId),
    });
    if (current.subscriberType === 'none') {
      // check uniq userName

      const userNameExist = await agencies.findOneBy({
        userName: dto.userName,
      });
      if (userNameExist) throw new HandledError(Dict.USERNAME_EXIST);

      await db.manager.update(UserEntity, { id: current.id }, {
        ...rest,
        city,
        areas,
        subscriberType: 'agency',
        type: 'AgencyEntity',
      } as any);
    } else {
      //insert
      const userNameExist = await agencies.findOneBy({
        id: Not(current.id),
        userName: dto.userName,
      });
      if (userNameExist) throw new HandledError(Dict.USERNAME_EXIST);

      const agency = await agencies.findOneBy({ id: current.id });
      agency.update({ ...rest, city, areas });
      await agency.save();
    }
    return Dict.SUCCESS_MESSAGE;
  }
}

export class UpsertAgencyUserRequestModel {
  @IsString()
  @ApiProperty()
  title: string;

  @IsString()
  @ApiProperty()
  cityCode: string;

  @IsString()
  @ApiProperty()
  phoneNumber: string;

  @IsString()
  @ApiProperty()
  licenseNumber: string;

  @IsString()
  @ApiProperty()
  licenseImg: string;

  @IsString()
  @ApiProperty()
  firstName: string;

  @ApiProperty({ type: LocationAddressModel })
  address: LocationAddressModel;

  @IsString()
  @ApiProperty()
  lastName: string;

  @IsString()
  @ApiProperty()
  userName: string;

  @IsString()
  @ApiProperty()
  cityId: string;

  @IsArray()
  @IsOptional()
  @ApiProperty()
  areasId: string[];
}
