import { ApiProperty } from '@nestjs/swagger';
import { CommandHandler, ICommandHandler } from '@nestjs/cqrs';
import { db } from '~db/repositories';
import { HandledError } from '~core/classes/handled-error';
import Dict from '~constants/dict';
import { UserEntity } from '~db/entities/user.entity';
import { PersonEntity } from '~db/entities/user/person.entity';
import { EntityManager, Not } from 'typeorm';
import ds from '~db/ds';
import { dot } from 'node:test/reporters';
import { IsNotEmpty, IsString } from 'class-validator';

const persons = db.getRepository(PersonEntity);
const users = db.getRepository(UserEntity);

export class UpsertPersonalUserCommand {
  constructor(
    public dto: UpsertPersonalUserRequestModel,
    public user?: UserEntity
  ) {}
}

@CommandHandler(UpsertPersonalUserCommand)
export class UpsertPersonalUserHandler
  implements ICommandHandler<UpsertPersonalUserCommand>
{
  constructor() {}
  async execute(command: UpsertPersonalUserCommand): Promise<any> {
    const current = command.user;

    if (current.subscriberType === 'none') {
      const userNameExist = await persons.findOneBy({
        userName: command.dto.userName,
      });
      if (userNameExist) throw new HandledError(Dict.USERNAME_EXIST);

      await ds.manager.update(UserEntity, { id: current.id }, {
        ...command.dto,
        subscriberType: 'personal',
        type: 'PersonEntity',
      } as any);
    } else {
      const userNameExist = await persons.findOneBy({
        id: Not(current.id),
        userName: command.dto.userName,
      });
      if (userNameExist) throw new HandledError(Dict.USERNAME_EXIST);

      const person = await persons.findOneBy({ id: current.id });
      person.update(command.dto);
      await person.save();
    }
    return Dict.SUCCESS_MESSAGE;
  }
}

export class UpsertPersonalUserRequestModel {
  @IsNotEmpty()
  @IsString()
  @ApiProperty()
  firstName: string;

  @IsNotEmpty()
  @IsString()
  @ApiProperty()
  lastName: string;

  @IsNotEmpty()
  @IsString()
  @ApiProperty()
  userName: string;

  @IsNotEmpty()
  @IsString()
  @ApiProperty()
  nationalCode: string;

  @IsNotEmpty()
  @IsString()
  @ApiProperty()
  nationalCardImg: string;
}
